using System.Collections.Generic;
using UnityEngine;
using UnityEditor;
using System.IO;
using Random = UnityEngine.Random;

public class MapToolsWindow : EditorWindow
{
    private SceneAsset sceneAsset;
    private DefaultAsset resourceFolder;
    private Vector2 logScrollPos;
    private List<LogEntry> logEntries = new List<LogEntry>();

    private string nameMapRu = "";
    private string nameMapEn = "";
    private string descriptionRu = "";
    private string descriptionEn = "";
    private string author = "";
    private string fileScreenshotName = "";
    private bool mapForPowerfulPhone;
    private string minimalVersionGame = "";
    private string mapVersion = "";

    private class LogEntry
    {
        public string Message;
        public LogType Type;
    }

    [MenuItem("Map Tools/Map Tools Window")]
    public static void ShowWindow()
    {
        MapToolsWindow window = GetWindow<MapToolsWindow>("Map Tools Window");
        window.minSize = new Vector2(600, 720);
    }

    private void OnEnable()
    {
        MapToolsApi.LoadMaps();
        if (MapToolsApi.Maps.Count == 0)
            MapToolsApi.AddMap();
        SynchronizeFieldsFromSelectedMap();
        Application.logMessageReceived += HandleLog;
    }

    private void OnDisable()
    {
        SynchronizeSelectedMapFromFields();
        MapToolsApi.SaveMaps();
        Application.logMessageReceived -= HandleLog;
    }

    private void HandleLog(string logString, string stackTrace, LogType type)
    {
        logEntries.Add(new LogEntry { Message = logString, Type = type });
        if (logEntries.Count > 10)
            logEntries.RemoveAt(0);
        Repaint();
    }

    private void DrawHorizontalLine(Color color)
    {
        Rect rect = GUILayoutUtility.GetRect(10, 5, GUILayout.ExpandWidth(true));
        EditorGUI.DrawRect(rect, color);
    }

    private void OnGUI()
    {
        GUILayout.BeginHorizontal();
        GUILayout.FlexibleSpace();
        GUILayout.Label("MapToolsApi v1.3", EditorStyles.label);
        GUILayout.EndHorizontal();
        GUILayout.Space(5);
        EditorGUI.BeginChangeCheck();
        DrawMultiMapSection();
        GUILayout.Space(10);
        DrawHorizontalLine(Color.gray);
        GUILayout.Space(20);
        DrawAssignMenu();
        GUILayout.Space(20);
        DrawMapSettings();
        GUILayout.Space(20);
        DrawHorizontalLine(Color.gray);
        GUILayout.Space(10);
        DrawButtons();
        GUILayout.Space(20);
        DrawLogArea();
        if (EditorGUI.EndChangeCheck())
        {
            SynchronizeSelectedMapFromFields();
            MapToolsApi.SaveMaps();
        }
    }


    private void DrawAssignMenu()
    {
        GUILayout.Label("Select a Scene and a Resource Folder", EditorStyles.boldLabel);
        GUILayout.Space(5);
        EditorGUI.BeginChangeCheck();
        SceneAsset newSceneAsset = EditorGUILayout.ObjectField("Scene", sceneAsset, typeof(SceneAsset), false) as SceneAsset;
        if (EditorGUI.EndChangeCheck())
        {
            MapConfiguration config = MapToolsApi.GetCurrentMap();
            string newPath = AssetDatabase.GetAssetPath(newSceneAsset);
            config.ScenePath = newPath;
            sceneAsset = newSceneAsset;
            SynchronizeSelectedMapFromFields();
        }

        EditorGUI.BeginChangeCheck();
        DefaultAsset newResourceFolder = EditorGUILayout.ObjectField("Resource Folder", resourceFolder, typeof(DefaultAsset), false) as DefaultAsset;
        if (EditorGUI.EndChangeCheck())
        {
            MapConfiguration config = MapToolsApi.GetCurrentMap();
            string newFolderPath = AssetDatabase.GetAssetPath(newResourceFolder);
            config.ResourceFolderPath = newFolderPath;
            resourceFolder = newResourceFolder;
            SynchronizeSelectedMapFromFields();
        }
    }

    private void DrawMapSettings()
    {
        GUILayout.Label("Map Settings", EditorStyles.boldLabel);
        GUILayout.Space(5);
        EditorGUI.indentLevel++;
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("Map title in Russian", GUILayout.Width(180));
        nameMapRu = EditorGUILayout.TextField(nameMapRu);
        EditorGUILayout.EndHorizontal();
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("Map title in English", GUILayout.Width(180));
        nameMapEn = EditorGUILayout.TextField(nameMapEn);
        EditorGUILayout.EndHorizontal();
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("Description in Russian", GUILayout.Width(180));
        descriptionRu = EditorGUILayout.TextField(descriptionRu);
        EditorGUILayout.EndHorizontal();
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("Description in English", GUILayout.Width(180));
        descriptionEn = EditorGUILayout.TextField(descriptionEn);
        EditorGUILayout.EndHorizontal();
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("Map author", GUILayout.Width(180));
        author = EditorGUILayout.TextField(author);
        EditorGUILayout.EndHorizontal();
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("Screenshot Path", GUILayout.Width(180));
        fileScreenshotName = EditorGUILayout.TextField(fileScreenshotName);
        if (GUILayout.Button("Select File", GUILayout.Width(100)))
        {
            string selectedFile = EditorUtility.OpenFilePanel("Select Screenshot File", "", "png,jpg");
            if (!string.IsNullOrEmpty(selectedFile))
                fileScreenshotName = Path.GetFullPath(selectedFile);
        }

        EditorGUILayout.EndHorizontal();
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("A map for a powerful phone", GUILayout.Width(180));
        mapForPowerfulPhone = EditorGUILayout.Toggle(mapForPowerfulPhone);
        EditorGUILayout.EndHorizontal();
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("Minimum version of the game", GUILayout.Width(180));
        minimalVersionGame = EditorGUILayout.TextField(minimalVersionGame);
        EditorGUILayout.EndHorizontal();
        EditorGUILayout.BeginHorizontal();
        EditorGUILayout.LabelField("Map version", GUILayout.Width(180));
        mapVersion = EditorGUILayout.TextField(mapVersion);
        EditorGUILayout.EndHorizontal();
        EditorGUI.indentLevel--;
    }

    private void DrawButtons()
    {
        GUILayout.Label("Build Asset Bundles", EditorStyles.boldLabel);
        GUILayout.Space(5);
        EditorGUILayout.BeginHorizontal();

        if (GUILayout.Button("Build for iOS", GUILayout.Height(50), GUILayout.ExpandWidth(true)))
        {
            EditorApplication.delayCall += () =>
            {
                SynchronizeSelectedMapFromFields();
                MapToolsApi.SaveMaps();
                MapToolsApi.BuildAllAssetBundlesIos();
            };
        }

        if (GUILayout.Button("Build for Android", GUILayout.Height(50), GUILayout.ExpandWidth(true)))
        {
            EditorApplication.delayCall += () =>
            {
                SynchronizeSelectedMapFromFields();
                MapToolsApi.SaveMaps();
                MapToolsApi.BuildAllAssetBundlesAndroid();
            };
        }

        EditorGUILayout.EndHorizontal();
    }

    private void DrawLogArea()
    {
        logScrollPos = EditorGUILayout.BeginScrollView(logScrollPos, GUILayout.Height(200));
        foreach (LogEntry entry in logEntries)
        {
            GUIStyle style = new GUIStyle(EditorStyles.label);
            switch (entry.Type)
            {
                case LogType.Error:
                case LogType.Exception:
                    style.normal.textColor = Color.red;
                    break;
                case LogType.Warning:
                    style.normal.textColor = Color.yellow;
                    break;
                default:
                    style.normal.textColor = Color.white;
                    break;
            }

            GUILayout.Label("[" + entry.Type.ToString() + "] " + entry.Message, style);
        }

        EditorGUILayout.EndScrollView();
    }


    private void DrawMultiMapSection()
    {
        GUILayout.Label("Select map", EditorStyles.boldLabel);
        GUILayout.Space(5);
        EditorGUILayout.BeginHorizontal();
        if (GUILayout.Button("Add Map", GUILayout.Width(100)))
        {
            MapToolsApi.AddMap();
            SynchronizeFieldsFromSelectedMap();
        }

        if (GUILayout.Button("Delete Map", GUILayout.Width(100)))
        {
            MapToolsApi.Maps.RemoveAt(MapToolsApi.SelectedMapIndex);
            if (MapToolsApi.Maps.Count == 0)
            {
                MapToolsApi.AddMap();
            }

            if (MapToolsApi.SelectedMapIndex >= MapToolsApi.Maps.Count)
                MapToolsApi.SelectedMapIndex = MapToolsApi.Maps.Count - 1;
            MapToolsApi.SaveMaps();
            SynchronizeFieldsFromSelectedMap();
        }

        if (MapToolsApi.Maps.Count > 0)
        {
            string[] mapNames = new string[MapToolsApi.Maps.Count];
            for (int i = 0; i < MapToolsApi.Maps.Count; i++)
                mapNames[i] = MapToolsApi.Maps[i].MapTitleEn;
            int newIndex = EditorGUILayout.Popup(MapToolsApi.SelectedMapIndex, mapNames);
            if (newIndex != MapToolsApi.SelectedMapIndex)
            {
                MapToolsApi.SelectedMapIndex = newIndex;
                SynchronizeFieldsFromSelectedMap();
            }
        }

        EditorGUILayout.EndHorizontal();
    }

    private void SynchronizeFieldsFromSelectedMap()
    {
        MapConfiguration config = MapToolsApi.GetCurrentMap();
        if (config != null)
        {
            nameMapRu = config.MapTitleRu;
            nameMapEn = config.MapTitleEn;
            descriptionRu = config.DescriptionRu;
            descriptionEn = config.DescriptionEn;
            author = config.Author;
            fileScreenshotName = config.ScreenshotPath;
            mapForPowerfulPhone = config.IsForPowerfulPhone;
            minimalVersionGame = config.MinimalGameVersion;
            mapVersion = config.MapVersion;
            sceneAsset = !string.IsNullOrEmpty(config.ScenePath) ? AssetDatabase.LoadAssetAtPath<SceneAsset>(config.ScenePath) : null;
            resourceFolder = !string.IsNullOrEmpty(config.ResourceFolderPath) ? AssetDatabase.LoadAssetAtPath<DefaultAsset>(config.ResourceFolderPath) : null;
        }
    }

    private void SynchronizeSelectedMapFromFields()
    {
        MapConfiguration config = MapToolsApi.GetCurrentMap();
        if (config != null)
        {
            bool isWrongName = string.IsNullOrWhiteSpace(nameMapEn);
            if (isWrongName)
            {
                Debug.LogWarning("A map name is empty.");
                nameMapEn = "Unknown" + Random.Range(1000, 99999);
            }

            config.MapTitleRu = nameMapRu;
            config.MapTitleEn = nameMapEn;
            config.DescriptionRu = descriptionRu;
            config.DescriptionEn = descriptionEn;
            config.Author = author;
            config.ScreenshotPath = fileScreenshotName;
            config.IsForPowerfulPhone = mapForPowerfulPhone;
            config.MinimalGameVersion = minimalVersionGame;
            config.MapVersion = mapVersion;
            config.ScenePath = sceneAsset != null ? AssetDatabase.GetAssetPath(sceneAsset) : "";
            config.ResourceFolderPath = resourceFolder != null ? AssetDatabase.GetAssetPath(resourceFolder) : "";
        }
    }
}